/* =========================
 * game.js  — フル版（カウントダウン対応）
 * ========================= */

/* ====== 設定 ====== */
const PLAY_TIME = 60;           // 秒
const MISS_LIMIT = 5;
const WORDS = ["AI", "スマホ", "だれでも", "プログラマー"];
const SPEED = { bg: 80, enemy: 120, bullet: 520, ebullet: 220, float: 45 };
const SPAWN = { enemy: [700, 1200], fire: 230 };
const SCORE_PER = 100;

/* ====== 要素取得 ====== */
const root   = document.getElementById('gameRoot');
const canvas = document.getElementById('game');
const ctx    = canvas.getContext('2d');
const elTime = document.getElementById('time');
const elScore= document.getElementById('score');
const elHigh = document.getElementById('high');
const elMiss = document.getElementById('miss');
const overEl = document.getElementById('over');
const ovTitle= document.getElementById('ovTitle');
const ovScore= document.getElementById('ovScore');
const ovHigh = document.getElementById('ovHigh');
const copyEl = document.getElementById('copy');
const ctaEl  = document.getElementById('cta');

// 敵弾の点滅（1秒あたりの点滅回数）
const E_BULLET_BLINK_HZ = 8;
let ENEMY_BULLET_SCALE = 1;
let FLOAT_FONT_SIZE = 24;

// 追加：ボス設定
const BOSS = {
  spawnElapsed: 30,   // 経過30秒で出現（＝残り30秒）
  baseSize: 180,      // 基準サイズ（画面スケールで拡縮）
  hp: 2500,           // まあまあ固め
  dmgPerHit: 30,      // 自弾1発のダメージ
  flash: 0.12,        // 被弾時の赤フラッシュ秒
  speedX: 80,         // 侵入時の横速度
  speedY: 120,        // 縦のふらつき速度
  fireEvery: [700, 1100], // 発射間隔ms（ランダム）
  spread: 12 * Math.PI/180 // 弾のばらつき（ラジアン）
};

// 追加：アイテムの吸引パラメータ
const ITEM = {
  seekDelay: 0.4,     // 生成から何秒後に吸引開始するか
  magnetRadius: 160,  // この距離以内に入ったら即吸引
  seekAccel: 320,     // プレイヤー方向への加速度
  maxSpeed: 360       // アイテムの最高速度（暴走防止）
};

/* カウントダウン要素 */
const cdLayer = document.querySelector('.countdown');
const cdSpan  = cdLayer ? cdLayer.querySelector('span') : null;

/* ====== アセット ====== */
function loadImg(src){ const i=new Image(); i.src=src; return i; }
const imgBG  = loadImg('img/background.webp');
const imgCat = loadImg('img/nyan.webp');
const imgCatIdle  = loadImg('img/cat_idle.webp');
const imgCatWalk1 = loadImg('img/cat_walk1.webp');
const imgCatWalk2 = loadImg('img/cat_walk2.webp');
const imgCatAtk   = loadImg('img/cat_attack.webp'); // 口開き等
const imgCatHurt  = loadImg('img/cat_hurt.webp');   // ダメージ時
const imgBug = loadImg('img/bug.webp');
const imgUp  = loadImg('img/atup.webp');

/* ====== 画面・スケール ====== */
let W=0, H=0, DPR=1;
let ENEMY_SIZE = 48;
let BULLET_FONT_BASE = 18;
let SPEED_SCALED = { ...SPEED };

// 追加：ボス関連
let boss = null;
let bossSpawned = false;
let BOSS_SIZE = 180;  // applyScale() で更新


function clamp(v, a, b){ return Math.max(a, Math.min(b, v)); }

function applyScale(){
  const S = clamp(H / 540, 0.8, 1.8);
  let scaleCat = 1.0;
  if (/Mobi|Android/i.test(navigator.userAgent)) {
    scaleCat = 0.60;   // ← スマホのときだけ 75% サイズに
    ENEMY_BULLET_SCALE = 0.20; 
    BULLET_FONT_BASE = Math.round(14 * S*0.8 );
    HUD_FONT_SIZE = Math.round(18 * S);  // ← HUD用スケールフォント
    FLOAT_FONT_SIZE = Math.round(24 * S); // ← フローティング文字用
  }
  player.w = Math.round(56 * S * scaleCat);
  player.h = Math.round(56 * S * scaleCat);
  ENEMY_SIZE = Math.round(48 * S);
  player.speed = 420 * S;
  SPEED_SCALED = {
    bg:      SPEED.bg * S,
    enemy:   SPEED.enemy * S,
    bullet:  SPEED.bullet * S,
    ebullet: SPEED.ebullet * S,
    float:   SPEED.float * S,
  };
  BOSS_SIZE = Math.round(BOSS.baseSize * S); //ボスサイズ
 
}

function fit(){
  const r = root.getBoundingClientRect();
  DPR = Math.min(2, window.devicePixelRatio || 1);
  canvas.width  = Math.round(r.width  * DPR);
  canvas.height = Math.round(r.height * DPR);
  canvas.style.width  = r.width  + 'px';
  canvas.style.height = r.height + 'px';
  W = r.width; H = r.height;
  ctx.setTransform(DPR,0,0,DPR,0,0);
  applyScale();
}

/* ====== 状態 ====== */
let state='boot';
let tLeft=PLAY_TIME, score=0, hi=Number(localStorage.getItem('mcp_game_hi')||0), miss=0;
//elHigh.textContent = hi.toString().padStart(4,'0');

const player = { x: 32, y: 140, w: 56, h: 56, power: 1, speed: 420,
   attackT: 0,   // 攻撃表示残り秒
   hurtT: 0,     // 被弾表示残り秒
   _py: 140,     // 前フレームY（移動判定用）
   moving: false // 歩きアニメ用フラグ
};
const bullets=[], enemies=[], ebullets=[], items=[], floats=[];
let bgX=0;

let last=0, fireTimer=0, enemyTimer=0;
let pointerY = null;
let keys = {};

/* ====== 共通ユーティリティ ====== */
function rnd(a,b){ return a + Math.random()*(b-a); }
function rects(a,b){
  return a.x < b.x + b.w && b.x < a.x + (a.w||0) &&
         a.y < b.y + b.h && b.y < a.y + (a.h||0);
}
function pad4(n){ return n.toString().padStart(4, '0'); }

function setState(s){
  state = s;
  if (s==='ready'){
    tLeft = PLAY_TIME;
    clearTimeout(enemyTimer);
    clearTimeout(fireTimer);
    score = 0;
    miss  = 0;
    player.power = 1;
    bullets.length = enemies.length = ebullets.length = items.length = floats.length = 0;
    overEl.hidden = true;
    nextEnemy();
    nextFire();
    // ★ ボス関係の初期化
    boss = null;
    bossSpawned = false;
    // （任意）背景スクロールも戻す
    bgX = 0;
  }
}

/* ====== スポーン／タイマ ====== */
function spawnEnemy(){
  const y = Math.max(20, Math.min(H-20-ENEMY_SIZE, rnd(20, H-40)));
  const e = { x: W + 50, y, w: ENEMY_SIZE, h: ENEMY_SIZE, t:0, nextShot: 400 + Math.random()*600 };
  enemies.push(e);
}
function nextEnemy(){
  clearTimeout(enemyTimer);
  const wait = rnd(...SPAWN.enemy);
  enemyTimer = setTimeout(()=>{ spawnEnemy(); nextEnemy(); }, wait);
}
function fire(){
  const text = WORDS[Math.floor(Math.random()*WORDS.length)];
  const b = {
    x: player.x + player.w * 0.6,
    y: player.y + player.h * 0.5,
    text,
    w: 0, h: 0,
    size: BULLET_FONT_BASE * player.power
  };
  bullets.push(b);
  // 攻撃アニメ：短時間だけ口開き
  player.attackT = 0.18;
}
function nextFire(){
  clearTimeout(fireTimer);
  fireTimer = setTimeout(()=>{ fire(); nextFire(); }, SPAWN.fire);
}

function spawnBoss(){
  boss = {
    x: W + 60,
    y: Math.max(30, Math.min(H - 30 - BOSS_SIZE, H * 0.25)),
    w: BOSS_SIZE,
    h: BOSS_SIZE,
    hp: BOSS.hp,
    maxHp: BOSS.hp,
    flash: 0,             // 被弾フラッシュ残時間
    entering: true,       // 侵入中フラグ
    fireMs: 600           // 次弾までの残りms
  };
  bossSpawned = true;

  // 通常敵は以降なるべく減らしたいならスポーン止める（任意）
  clearTimeout(enemyTimer);
}

function dropItem(x, y){
  items.push({
    x, y,
    w: Math.round(32 * (H/540)), h: Math.round(32 * (H/540)),
    vx: -SPEED_SCALED.enemy * 0.6,
    vy: -80 * (H/540),
    life: 6,
    t: 0 
  });
}
function addFloat(x,y,txt,color='#fff'){
  floats.push({ x, y, t:0, txt, color });
}

/* ====== 入力 ====== */
function onPointer(e){
 // スワイプでページがスクロールしないよう抑止
 if (e.cancelable) e.preventDefault();
 const rect = canvas.getBoundingClientRect();
 // スマホの場合：指の座標（スワイプ位置）をそのまま追従させる
 if (e.touches) {
   pointerY = e.touches[0].clientY - rect.top;
 } else {
   // PCの場合は従来どおりマウス位置で追従
   pointerY = e.clientY - rect.top;
 }
}
function onPointerUp(){ pointerY = null; }
function onKey(e, down){
  if (e.code==='Space') { if (down) fire(); e.preventDefault(); }
  if (e.code==='ArrowUp')   keys.up = down;
  if (e.code==='ArrowDown') keys.down = down;
}

/* ====== HUD ====== */
function updHUD(){
  elTime.textContent  = tLeft.toFixed(1).padStart(4,'0');
  elScore.textContent = pad4(score);
  // 残りHPをハートで表示
  const hearts = MISS_LIMIT - miss;
  elMiss.textContent = '❤'.repeat(hearts);
}

/* ====== メインループ ====== */
function update(dt){
  if (boss && enemies.length > 0) enemies.length = 0; // 既存敵を掃除
  if (state!=='ready') return;

  // 時間
  tLeft -= dt;
  if (tLeft <= 0) { endGame('TIME UP'); return; }

  const elapsed = PLAY_TIME - tLeft;
  if (!bossSpawned && elapsed >= BOSS.spawnElapsed) {
    spawnBoss();
  }

  // 背景
  bgX -= SPEED_SCALED.bg * dt;
  if (bgX < -W) bgX += W;

  // 入力による移動（マウス/タッチは追従、キーボードは速度移動）
  if (pointerY != null){
    // 追従（なめらか移動）
    const target = pointerY - player.h * 0.5;
    player.y += (target - player.y) * Math.min(1, dt*10);
  } else {
    if (keys.up)   player.y -= player.speed * dt;
    if (keys.down) player.y += player.speed * dt;
  }
  player.y = clamp(player.y, 0, H - player.h);
  // 歩き判定（前フレームと比較）
  player.moving = Math.abs(player.y - player._py) > 0.5;
  player._py = player.y;
  // アニメ用タイマー更新
  if (player.attackT > 0) player.attackT = Math.max(0, player.attackT - dt);
  if (player.hurtT   > 0) player.hurtT   = Math.max(0, player.hurtT   - dt);

  // 敵の移動・射撃
  for (let i=enemies.length-1;i>=0;i--){
    const e = enemies[i];
    e.x -= SPEED_SCALED.enemy * dt;
    e.t += dt*1000;
    e.nextShot -= dt*1000;
    if (e.nextShot <= 0){
      // 敵弾（プレイヤーへ）
      const cx = e.x, cy = e.y + e.h*0.5;
      const px = player.x + player.w*0.5, py = player.y + player.h*0.5;
      const ang = Math.atan2(py - cy, px - cx);
      const v = SPEED_SCALED.ebullet;
      ebullets.push({ x:cx, y:cy, vx:Math.cos(ang)*v, vy:Math.sin(ang)*v,w: 20 * ENEMY_BULLET_SCALE, h: 20  * ENEMY_BULLET_SCALE, t:0 });
      e.nextShot = 400 + Math.random()*800;
    }

    // 画面外で消す
    if (e.x + e.w < -40){
      enemies.splice(i,1);
      // 取り逃し扱い (ミス)
      miss++;
      addFloat(20, 30, 'MISS', '#f55');
      if (miss >= MISS_LIMIT){ endGame('MISS'); return; }
    }
  }

      // --- ボスがいるときの処理 ---
    if (boss) {
      // 侵入中：右端から所定位置（画面右からw+40手前）まで移動
      const targetX = Math.max(40, W - boss.w - 40);
      if (boss.entering) {
        boss.x -= BOSS.speedX * dt;
        if (boss.x <= targetX) {
          boss.x = targetX;
          boss.entering = false;
        }
      } else {
        // 画面内で上下にふらつく
        boss.y += BOSS.speedY * dt * (Math.sin(performance.now()/600) );
        boss.y = clamp(boss.y, 20, H - 20 - boss.h);
      }

      // 射撃（プレイヤーへ狙い撃ち＋少しばらす）
      boss.fireMs -= dt * 1000;
      if (boss.fireMs <= 0) {
        const px = player.x + player.w*0.5;
        const py = player.y + player.h*0.5;
        const cx = boss.x, cy = boss.y + boss.h*0.5;
        const baseAng = Math.atan2(py - cy, px - cx);
        // 3連弾（中央＋左右にスプレッド）
        const v = SPEED_SCALED.ebullet * 1.1;
        [0, -BOSS.spread, +BOSS.spread].forEach(d=>{
          const a = baseAng + d;
            ebullets.push({ x:cx, y:cy, vx:Math.cos(a)*v, vy:Math.sin(a)*v,  w: 20 * ENEMY_BULLET_SCALE, h: 20  * ENEMY_BULLET_SCALE, t:0 });
        });
        boss.fireMs = rnd(...BOSS.fireEvery);
      }

      // フラッシュ減衰
      if (boss.flash > 0) boss.flash = Math.max(0, boss.flash - dt);
    }

  // 自弾（テキスト）の移動・当たり
  for (let i=bullets.length-1;i>=0;i--){
    const b = bullets[i];
    b.x += SPEED_SCALED.bullet * dt;

    // テキストの矩形（描画前に推定）
    ctx.font =`bold ${FLOAT_FONT_SIZE}px/1 sans-serif`;
    const m = ctx.measureText(b.text);
    b.w = m.width;
    b.h = b.size; // 粗め

    // 敵に命中？
    for (let j=enemies.length-1;j>=0;j--){
      const e = enemies[j];
      if (rects({x:b.x, y:b.y-b.h, w:b.w, h:b.h}, e)){
        enemies.splice(j,1);
        bullets.splice(i,1);
        // スコア & ドロップ
        score += SCORE_PER;
        if (Math.random() < 0.5){
          dropItem(e.x, e.y);
        }
        addFloat(e.x, e.y, '+100', '#6f6');
        break;
      }
      // --- ボスに命中？（敵ヒット処理の後に） ---
      if (boss) {
        // 自弾bの矩形は {x:b.x, y:b.y-b.h, w:b.w, h:b.h}
        if (rects({x:b.x, y:b.y-b.h, w:b.w, h:b.h}, boss)) {
          bullets.splice(i,1);
          boss.hp -= BOSS.dmgPerHit;
          boss.flash = BOSS.flash;
          addFloat(boss.x + boss.w*0.5, boss.y, `-${BOSS.dmgPerHit}`, '#faa');
        }
      }
    }

    if (boss && i < bullets.length) {
      if (rects({x:b.x, y:b.y-b.h, w:b.w, h:b.h}, boss)) {
        bullets.splice(i,1);
        boss.hp -= BOSS.dmgPerHit;
        boss.flash = BOSS.flash;
        addFloat(boss.x + boss.w*0.5, boss.y, `-${BOSS.dmgPerHit}`, '#faa');

        if (boss.hp <= 0) {
          score += 1000;
          addFloat(boss.x + boss.w*0.5, boss.y, 'BOSS DOWN +1000', '#6ff');
          boss = null;
          nextEnemy(); // 通常敵の湧き再開
        }
      }
    }

    // 画面外
    if (i < bullets.length && b.x > W + 60){
      bullets.splice(i,1);
    }
  }

  // 敵弾
  for (let i=ebullets.length-1;i>=0;i--){
    const b = ebullets[i];
    b.t = (b.t || 0) + dt;   // ← 点滅用の経過時間
    b.x += b.vx * dt;
    b.y += b.vy * dt;

    // プレイヤーに命中（※ 猫は固定サイズ。power を掛けない）
    if (rects(b, { x: player.x, y: player.y, w: player.w, h: player.h })){
      ebullets.splice(i,1);
      miss++;
      player.hurtT = 0.25; // 被弾アニメ
      addFloat(player.x, player.y, 'HIT', '#f66');
      if (miss >= MISS_LIMIT){ endGame('MISS'); return; }
      continue;
    }
    // 画面外
    if (b.x<-50||b.y<-50||b.x>W+50||b.y>H+50){
      ebullets.splice(i,1);
    }
  }

  // アイテム
  // アイテム
for (let i = items.length - 1; i >= 0; i--) {
  const it = items[i];

  // 経過時間
  it.t += dt;

  // ---- 吸引（マグネット）ロジック ----
  // プレイヤー中心へ向かうベクトル
  const px = player.x + player.w * 0.5;
  const py = player.y + player.h * 0.5;
  const ix = it.x + it.w * 0.5;
  const iy = it.y + it.h * 0.5;
  let dx = px - ix;
  let dy = py - iy;
  const dist = Math.hypot(dx, dy) || 1;

  // 「一定時間経過」または「一定距離以内」で吸引開始
  if (it.t > ITEM.seekDelay || dist < ITEM.magnetRadius) {
    dx /= dist; // 正規化
    dy /= dist;
    // プレイヤー方向へ加速
    it.vx += dx * ITEM.seekAccel * dt;
    it.vy += dy * ITEM.seekAccel * dt;

    // 最高速度を制限
    const v = Math.hypot(it.vx, it.vy);
    if (v > ITEM.maxSpeed) {
      it.vx = (it.vx / v) * ITEM.maxSpeed;
      it.vy = (it.vy / v) * ITEM.maxSpeed;
    }
  } else {
    // 吸引前は従来のふるまい（軽い重力）
    it.vy += 50 * dt;
  }

  // 位置更新
  it.x += it.vx * dt;
  it.y += it.vy * dt;

  // 取得判定（猫サイズは固定）
 if (rects(it, { x: player.x, y: player.y, w: player.w, h: player.h })) {
  items.splice(i, 1);
  player.power = Math.min(3, player.power + 0.5);
  addFloat(player.x, player.y, 'POWER UP!', '#6cf');
  score += 50; // アイテム取得ボーナス
  addFloat(player.x, player.y - 20, '+50', '#ff6');
  continue;
}

  // 画面外や寿命で消滅
  if (it.life <= 0 || it.x + it.w < -40 || it.y > H + 40) {
    items.splice(i, 1);
  } else {
    it.life -= dt;
  }
}


  // フローティングテキスト
  for (let i=floats.length-1;i>=0;i--){
    const f = floats[i];
    f.t += dt;
    f.y -= SPEED_SCALED.float * dt;
    if (f.t > 1.2) floats.splice(i,1);
  }

  // HUD
  updHUD();
}
function draw(){
  // 背景（単純スクロール。背景画像がW以上ある前提）
  if (imgBG.complete){
    const x1 = Math.floor(bgX % W);
    ctx.drawImage(imgBG, x1, 0, W, H);
    ctx.drawImage(imgBG, x1 + W, 0, W, H);
  }else{
    ctx.fillStyle = '#223';
    ctx.fillRect(0,0,W,H);
  }

  // 敵
  enemies.forEach(e=>{
    if (imgBug.complete) ctx.drawImage(imgBug, e.x, e.y, e.w, e.h);
    else { ctx.fillStyle='#9f3'; ctx.fillRect(e.x,e.y,e.w,e.h); }
  });

  // アイテム
  items.forEach(it=>{
    if (imgUp.complete) ctx.drawImage(imgUp, it.x, it.y, it.w, it.h);
    else { ctx.fillStyle='#0cf'; ctx.fillRect(it.x,it.y,it.w,it.h); }
  });

    // 敵弾（赤↔白点滅 & うっすら縁取り）
  ebullets.forEach(b=>{
    const phase = Math.floor((b.t || 0) * E_BULLET_BLINK_HZ) % 2; // 0 or 1
    ctx.fillStyle = phase ? '#fff' : '#f33';
    ctx.fillRect(b.x, b.y, b.w, b.h);
    // 視認性向上のための縁取り（任意）
    ctx.lineWidth = 0.5;
    ctx.strokeStyle = phase ? '#f33' : '#fff';
    ctx.strokeRect(b.x + 0.5, b.y + 0.5, b.w - 1, b.h - 1);
  });

  // 自弾（攻撃テキスト）— power で文字だけ大きく
  bullets.forEach(b=>{
    ctx.font = `bold ${b.size}px/1 sans-serif`;
    ctx.fillStyle = '#fff';
    ctx.textBaseline = 'middle';
    ctx.fillText(b.text, b.x, b.y);
  });

  // --- ボス描画 ---
  if (boss) {
    if (imgBug.complete) {
      ctx.drawImage(imgBug, boss.x, boss.y, boss.w, boss.h);
    } else {
      ctx.fillStyle = '#8f6';
      ctx.fillRect(boss.x, boss.y, boss.w, boss.h);
    }
    // 被弾フラッシュ（赤い被せ）
    if (boss.flash > 0) {
      ctx.globalAlpha = 0.4;
      ctx.fillStyle = '#f00';
      ctx.fillRect(boss.x, boss.y, boss.w, boss.h);
      ctx.globalAlpha = 1;
    }
  }

  // --- ボスHPバー ---
  if (boss) {
    const bw = boss.w;
    const bh = 10;
    const bx = boss.x;
    const by = boss.y + boss.h + 8; // ボスの下に8px離して表示
    const rate = Math.max(0, boss.hp / boss.maxHp);

    // 枠
    ctx.lineWidth = 2;
    ctx.strokeStyle = '#fff';
    ctx.strokeRect(bx, by, bw, bh);

    // 中身
    ctx.fillStyle = '#f44';
    ctx.fillRect(bx, by, bw * rate, bh);

    // ラベル
    ctx.font = 'bold 12px/1 sans-serif';
    ctx.fillStyle = '#fff';
    ctx.textAlign = 'center';
    ctx.fillText('BOSS', bx + bw/2, by - 4);
    ctx.textAlign = 'start';
  }


  // プレイヤー（アニメ）
  const nowMS = performance.now();
  let catImg = imgCat; // フォールバック
  if (player.hurtT > 0 && imgCatHurt.complete) {
    catImg = imgCatHurt;
  } else if (player.attackT > 0 && imgCatAtk.complete) {
    catImg = imgCatAtk;
  } else if (player.moving) {
    // 150ms周期で歩きコマ切り替え
    const phase = Math.floor(nowMS / 150) % 2;
    if (phase === 0 && imgCatWalk1.complete) catImg = imgCatWalk1;
    else if (imgCatWalk2.complete)            catImg = imgCatWalk2;
    else if (imgCatIdle.complete)             catImg = imgCatIdle;
  } else if (imgCatIdle.complete) {
    catImg = imgCatIdle;
  }
  ctx.drawImage(catImg, player.x, player.y, player.w, player.h);

  // フローティングテキスト
  floats.forEach(f=>{
    ctx.globalAlpha = Math.max(0, 1 - f.t/1.2);
    const isMobile = /Mobi|Android/i.test(navigator.userAgent);
    const size = isMobile ? 14 : 18;
    ctx.font = `bold ${size}px/1 sans-serif`;
    ctx.fillStyle = f.color || '#fff';
    ctx.fillText(f.txt, f.x, f.y);
    ctx.globalAlpha = 1;
  });
}

function loop(t){
  const now = t || performance.now();
  if (!last) last = now;
  const dt = Math.min(0.05, (now - last)/1000);
  last = now;
  update(dt);
  draw();
  requestAnimationFrame(loop);
}

/* ====== 終了処理 ====== */
function endGame(reason){
  setState('over');
  clearTimeout(enemyTimer);
  clearTimeout(fireTimer);
  if (score > hi){
    hi = score;
    localStorage.setItem('mcp_game_hi', String(hi));
  }
  if (reason==='TIME UP'){
    ovTitle.textContent = 'ゲームクリア！';
    // 残HPボーナス
    const bonus = (MISS_LIMIT - miss) * 100;
    score += bonus;
    addFloat(player.x, player.y, `CLEAR BONUS +${bonus}`, '#6ff');
  } else {
    ovTitle.textContent = 'やられた！';
  }
  ovScore.textContent = pad4(score);
  ovHigh.textContent  = pad4(hi);
  const msg =
    (score>=800)? "AI×スマホで本格ゲームも作れる！"
  : (score>=500)? "スマホで作って学んで、すぐプログラマー。"
  : (score>=200)? "AI×スマホでだれでもプログラマー"
  : "まずはタップで遊んで作るところから。";
  copyEl.textContent = msg;
  overEl.hidden = false;
  ctaEl.classList.remove('show');
  setTimeout(()=> ctaEl.classList.add('show'), 3000);
}

/* ====== カウントダウン開始 ====== */
function startWithCountdown(){
  if (!cdLayer || !cdSpan){
    setState('ready');
    last = 0;
    requestAnimationFrame(loop);
    return;
  }
  cdLayer.style.display = 'flex';
  cdLayer.classList.remove('fade');
  let n = 3;
  cdSpan.textContent = String(n);
  const tick = () => {
    if (n > 1){
      n--;
      cdLayer.classList.add('fade');
      setTimeout(()=>{
        cdSpan.textContent = String(n);
        cdLayer.classList.remove('fade');
      }, 300);
      setTimeout(tick, 800);
    }else{
      cdLayer.classList.add('fade');
      cdSpan.textContent = 'GO!';
      setTimeout(()=>{
        cdLayer.style.display = 'none';
        setState('ready');
        last = 0;
        requestAnimationFrame(loop);
      }, 500);
    }
  };
  setTimeout(tick, 800);
}

/* ====== 初期化 ====== */
function boot(){
  fit();
  overEl.hidden = true;
  updHUD();
  startWithCountdown();   // ← カウントダウン経由で開始
}

/* ====== イベント登録 ====== */
window.addEventListener('resize', fit, { passive:true });
canvas.addEventListener('pointerdown', onPointer, { passive:false });
canvas.addEventListener('pointermove', onPointer, { passive:false });
window.addEventListener('blur', onPointerUp, { passive:true });

canvas.addEventListener('touchmove', onPointer, { passive:false });
canvas.addEventListener('touchstart', onPointer, { passive:false });
canvas.addEventListener('touchend', onPointerUp, { passive:true });

window.addEventListener('keydown', (e)=>onKey(e,true));
window.addEventListener('keyup',   (e)=>onKey(e,false));

document.getElementById('btnStart')?.addEventListener('click', ()=>{
  overEl.hidden = true;
  startWithCountdown();   // ←修正
});
document.getElementById('btnAgain')?.addEventListener('click', ()=>{
  overEl.hidden = true;
  startWithCountdown();   // ←修正
});
document.getElementById('btnShare')?.addEventListener('click', ()=>{
  const text = `スコア ${score}！ #AI×スマホ`;
  const url  = location.href;
  const shareUrl = `https://twitter.com/intent/tweet?text=${encodeURIComponent(text)}&url=${encodeURIComponent(url)}`;
// 常に iframe を抜けてブラウザ全体で遷移させる
 window.top.location.href = shareUrl;
});

document.getElementById('btnRetry')?.addEventListener('click', ()=>{
  overEl.hidden = true;
  startWithCountdown();
});

/* ====== 初期HUD ====== */
elHigh.textContent = pad4(hi);
updHUD();

/* ====== 起動 ====== */
boot();